<?php
/**
 * Subscription Refresh Cron Job
 * Google Play aboneliklerini periyodik olarak doğrular
 *
 * cPanel Cron Ayarı:
 * Frequency: Once per day (00:00)
 * Command: /usr/bin/php /home/username/public_html/backend/cron/refresh_subscriptions.php
 */

// Set timezone
date_default_timezone_set('Europe/Istanbul');

// Load dependencies
require_once __DIR__ . '/../core/Database.php';
require_once __DIR__ . '/../config/app.php';

$config = require __DIR__ . '/../config/app.php';

echo "=== Subscription Refresh Cron Started ===\n";
echo "Time: " . date('Y-m-d H:i:s') . "\n\n";

try {
    // Aktif abonelikleri getir
    $subscriptions = Database::fetchAll(
        'SELECT * FROM subscriptions
         WHERE is_active = 1
         AND expiry_time > NOW()
         ORDER BY expiry_time ASC'
    );

    echo "Found " . count($subscriptions) . " active subscriptions\n";

    $refreshed = 0;
    $expired = 0;
    $errors = 0;

    foreach ($subscriptions as $sub) {
        echo "\n--- Subscription ID: {$sub['id']} (User: {$sub['user_id']}) ---\n";

        // Son doğrulama 6 saatten eski mi?
        $lastVerified = strtotime($sub['last_verified_at'] ?? '1970-01-01');
        $now = time();

        if (($now - $lastVerified) < 6 * 3600) {
            echo "Recently verified, skipping...\n";
            continue;
        }

        // Google Play'den doğrula
        $result = verifyWithGoogle(
            $config['google_play']['package_name'],
            $sub['product_id'],
            getPurchaseTokenFromHash($sub['purchase_token_hash'], $sub['user_id']),
            $config
        );

        if (!$result['success']) {
            echo "Verification failed: " . ($result['error'] ?? 'Unknown') . "\n";
            $errors++;

            // Log error
            Database::insert(
                'INSERT INTO subscription_logs (user_id, action, product_id, status, details)
                 VALUES (?, ?, ?, ?, ?)',
                [
                    $sub['user_id'], 'cron_verify_failed',
                    $sub['product_id'], 'error',
                    json_encode(['error' => $result['error'] ?? 'Unknown'])
                ]
            );
            continue;
        }

        $data = $result['data'];
        $expiryTime = date('Y-m-d H:i:s', ($data['expiryTimeMillis'] ?? 0) / 1000);
        $isActive = ($data['paymentState'] ?? 0) == 1
            && (($data['expiryTimeMillis'] ?? 0) / 1000) > $now;
        $autoRenewing = $data['autoRenewing'] ?? false;

        // Update database
        Database::execute(
            'UPDATE subscriptions SET
                is_active = ?,
                expiry_time = ?,
                auto_renewing = ?,
                last_verified_at = NOW(),
                raw_response = ?,
                updated_at = NOW()
             WHERE id = ?',
            [
                $isActive ? 1 : 0,
                $expiryTime,
                $autoRenewing ? 1 : 0,
                json_encode($data),
                $sub['id']
            ]
        );

        if ($isActive) {
            echo "✓ Refreshed - Expires: {$expiryTime}\n";
            $refreshed++;
        } else {
            echo "✗ Expired\n";
            $expired++;
        }

        // Log success
        Database::insert(
            'INSERT INTO subscription_logs (user_id, action, product_id, status)
             VALUES (?, ?, ?, ?)',
            [
                $sub['user_id'], 'cron_verify_success',
                $sub['product_id'], $isActive ? 'active' : 'expired'
            ]
        );

        // Rate limit: 1 request per second
        sleep(1);
    }

    echo "\n=== Summary ===\n";
    echo "Refreshed: {$refreshed}\n";
    echo "Expired: {$expired}\n";
    echo "Errors: {$errors}\n";
    echo "\n=== Cron Completed ===\n";

} catch (Exception $e) {
    echo "\n!!! FATAL ERROR !!!\n";
    echo $e->getMessage() . "\n";
    echo $e->getTraceAsString() . "\n";
    exit(1);
}

// ============================================
// Helper Functions
// ============================================

/**
 * Google Play Developer API doğrulama
 */
function verifyWithGoogle(string $packageName, string $productId, string $purchaseToken, array $config): array
{
    $serviceAccountFile = $config['google_play']['service_account_json'];

    if (!file_exists($serviceAccountFile)) {
        // Development mode
        if ($config['debug'] ?? false) {
            return [
                'success' => true,
                'data' => [
                    'startTimeMillis'  => time() * 1000,
                    'expiryTimeMillis' => (time() + 30 * 86400) * 1000,
                    'autoRenewing'     => true,
                    'paymentState'     => 1,
                ],
            ];
        }

        return ['success' => false, 'error' => 'Service account file not found'];
    }

    try {
        $accessToken = getGoogleAccessToken($serviceAccountFile);
        if (!$accessToken) {
            return ['success' => false, 'error' => 'Failed to get access token'];
        }

        $url = "https://androidpublisher.googleapis.com/androidpublisher/v3/applications/{$packageName}/purchases/subscriptions/{$productId}/tokens/{$purchaseToken}";

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER     => ["Authorization: Bearer {$accessToken}"],
            CURLOPT_TIMEOUT        => 30,
            CURLOPT_SSL_VERIFYPEER => true,
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpCode !== 200) {
            return ['success' => false, 'error' => "HTTP {$httpCode}", 'raw' => $response];
        }

        $data = json_decode($response, true);
        return ['success' => true, 'data' => $data];

    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Google Service Account OAuth2 token
 */
function getGoogleAccessToken(string $serviceAccountFile): ?string
{
    $sa = json_decode(file_get_contents($serviceAccountFile), true);
    if (!$sa || !isset($sa['client_email'], $sa['private_key'])) {
        return null;
    }

    $now = time();
    $header = base64_encode(json_encode(['alg' => 'RS256', 'typ' => 'JWT']));
    $claim = base64_encode(json_encode([
        'iss'   => $sa['client_email'],
        'scope' => 'https://www.googleapis.com/auth/androidpublisher',
        'aud'   => 'https://oauth2.googleapis.com/token',
        'iat'   => $now,
        'exp'   => $now + 3600,
    ]));

    $signatureInput = "{$header}.{$claim}";
    $privateKey = openssl_pkey_get_private($sa['private_key']);
    openssl_sign($signatureInput, $signature, $privateKey, OPENSSL_ALGO_SHA256);

    $jwt = $signatureInput . '.' . base64_encode($signature);

    $ch = curl_init('https://oauth2.googleapis.com/token');
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_POSTFIELDS     => http_build_query([
            'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
            'assertion'  => $jwt,
        ]),
    ]);

    $response = json_decode(curl_exec($ch), true);
    curl_close($ch);

    return $response['access_token'] ?? null;
}

/**
 * Purchase token'ı hash'ten geri getir
 * NOT: Güvenlik için token'lar hash olarak saklanır.
 * Cron job'da token tekrar doğrulanamaz, sadece mevcut DB bilgisi güncellenir.
 * Gerçek implementasyonda token'ı şifreleyip saklamak daha iyi olabilir.
 */
function getPurchaseTokenFromHash(string $hash, int $userId): string
{
    // Bu örnekte hash'i kullanıyoruz ancak gerçek token gerekli
    // Production'da token'ı encrypted olarak saklamalısınız
    // Şimdilik development için mock döndürüyoruz
    return 'mock_token_' . $hash;
}
