<?php
/**
 * JSON Response Helper
 */
class Response
{
    /**
     * JSON yanıt gönder
     */
    public static function json(mixed $data, int $status = 200, array $headers = []): void
    {
        http_response_code($status);
        header('Content-Type: application/json; charset=utf-8');

        foreach ($headers as $key => $value) {
            header("$key: $value");
        }

        echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        exit;
    }

    /**
     * Başarılı yanıt
     */
    public static function success(mixed $data = null, string $message = 'OK', int $status = 200): void
    {
        self::json([
            'success' => true,
            'message' => $message,
            'data'    => $data,
        ], $status);
    }

    /**
     * Hata yanıtı
     */
    public static function error(string $message, string $code = 'ERROR', int $status = 400, mixed $details = null): void
    {
        $response = [
            'success' => false,
            'error'   => [
                'code'    => $code,
                'message' => $message,
            ],
        ];
        if ($details !== null) {
            $response['error']['details'] = $details;
        }
        self::json($response, $status);
    }

    /**
     * Sayfalanmış yanıt
     */
    public static function paginated(array $items, int $total, int $page, int $limit): void
    {
        self::json([
            'success' => true,
            'data'    => $items,
            'meta'    => [
                'total'       => $total,
                'page'        => $page,
                'limit'       => $limit,
                'totalPages'  => (int)ceil($total / max($limit, 1)),
            ],
        ]);
    }

    /**
     * Dosya indirme yanıtı
     */
    public static function file(string $filePath, string $filename, string $mimeType = 'application/octet-stream'): void
    {
        if (!file_exists($filePath)) {
            self::error('Dosya bulunamadı', 'FILE_NOT_FOUND', 404);
        }

        header('Content-Type: ' . $mimeType);
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . filesize($filePath));
        header('Cache-Control: no-cache, must-revalidate');

        readfile($filePath);
        exit;
    }

    /**
     * Dosya inline serve (okuma için)
     */
    public static function serve(string $filePath, string $mimeType, int $cacheSeconds = 0): void
    {
        if (!file_exists($filePath)) {
            self::error('Dosya bulunamadı', 'FILE_NOT_FOUND', 404);
        }

        header('Content-Type: ' . $mimeType);
        header('Content-Length: ' . filesize($filePath));

        if ($cacheSeconds > 0) {
            header("Cache-Control: public, max-age={$cacheSeconds}");
        }

        readfile($filePath);
        exit;
    }
}
