<?php
/**
 * JWT (JSON Web Token) Helper
 * Harici kütüphane olmadan basit HMAC-SHA256 implementasyonu
 */
class JWT
{
    private static ?array $config = null;

    private static function getConfig(): array
    {
        if (self::$config === null) {
            $app = require __DIR__ . '/../config/app.php';
            self::$config = $app['jwt'];
        }
        return self::$config;
    }

    /**
     * Access token oluştur
     */
    public static function createAccessToken(array $payload): string
    {
        $config = self::getConfig();
        $payload['iat'] = time();
        $payload['exp'] = time() + $config['access_ttl'];
        $payload['type'] = 'access';
        return self::encode($payload, $config['secret']);
    }

    /**
     * Refresh token oluştur
     */
    public static function createRefreshToken(array $payload): string
    {
        $config = self::getConfig();
        $payload['iat'] = time();
        $payload['exp'] = time() + $config['refresh_ttl'];
        $payload['type'] = 'refresh';
        return self::encode($payload, $config['secret']);
    }

    /**
     * Token doğrula ve payload döndür
     */
    public static function verify(string $token, string $expectedType = 'access'): ?array
    {
        $config = self::getConfig();
        $payload = self::decode($token, $config['secret']);

        if ($payload === null) return null;
        if (($payload['exp'] ?? 0) < time()) return null;
        if (($payload['type'] ?? '') !== $expectedType) return null;

        return $payload;
    }

    /**
     * JWT encode
     */
    private static function encode(array $payload, string $secret): string
    {
        $header = self::base64UrlEncode(json_encode(['typ' => 'JWT', 'alg' => 'HS256']));
        $body = self::base64UrlEncode(json_encode($payload));
        $signature = self::base64UrlEncode(
            hash_hmac('sha256', "$header.$body", $secret, true)
        );
        return "$header.$body.$signature";
    }

    /**
     * JWT decode
     */
    private static function decode(string $token, string $secret): ?array
    {
        $parts = explode('.', $token);
        if (count($parts) !== 3) return null;

        [$header, $body, $signature] = $parts;

        $expectedSig = self::base64UrlEncode(
            hash_hmac('sha256', "$header.$body", $secret, true)
        );

        if (!hash_equals($expectedSig, $signature)) return null;

        $payload = json_decode(self::base64UrlDecode($body), true);
        return is_array($payload) ? $payload : null;
    }

    private static function base64UrlEncode(string $data): string
    {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }

    private static function base64UrlDecode(string $data): string
    {
        return base64_decode(strtr($data, '-_', '+/'));
    }
}
