<?php
/**
 * Progress Controller - Okuma ilerleme yönetimi
 */
class ProgressController
{
    /**
     * POST /api/progress/update
     */
    public function update(array $params): void
    {
        global $currentUser;

        $data = Request::validate(['bookId']);

        $bookId      = (int)$data['bookId'];
        $lang        = Request::input('lang', 'tr');
        $lastFormat  = Request::input('lastFormat', 'chunk');
        $lastPage    = (int)Request::input('lastPage', 0);
        $lastLocation= Request::input('lastLocation', null);
        $progressPercent = (float)Request::input('progressPercent', 0);
        $readingTime = (int)Request::input('readingTime', 0); // eklenen saniye

        // Kitap var mı kontrol
        $book = Database::fetch('SELECT id, total_pages FROM books WHERE id = ?', [$bookId]);
        if (!$book) {
            Response::error('Kitap bulunamadı', 'BOOK_NOT_FOUND', 404);
        }

        // Yüzde hesapla (gönderilmediyse)
        if ($progressPercent <= 0 && $book['total_pages'] > 0) {
            $progressPercent = min(100, round(($lastPage / $book['total_pages']) * 100, 2));
        }

        // Upsert
        $existing = Database::fetch(
            'SELECT id, total_reading_time FROM reading_progress WHERE user_id = ? AND book_id = ? AND lang = ?',
            [$currentUser['id'], $bookId, $lang]
        );

        if ($existing) {
            $totalTime = (int)$existing['total_reading_time'] + $readingTime;
            Database::execute(
                'UPDATE reading_progress SET
                    last_format = ?, last_page = ?, last_location = ?,
                    progress_percent = ?, total_reading_time = ?, updated_at = NOW()
                 WHERE id = ?',
                [$lastFormat, $lastPage, $lastLocation, $progressPercent, $totalTime, $existing['id']]
            );
        } else {
            Database::insert(
                'INSERT INTO reading_progress
                    (user_id, book_id, lang, last_format, last_page, last_location, progress_percent, total_reading_time)
                 VALUES (?, ?, ?, ?, ?, ?, ?, ?)',
                [$currentUser['id'], $bookId, $lang, $lastFormat, $lastPage, $lastLocation, $progressPercent, $readingTime]
            );
        }

        Response::success([
            'bookId'          => $bookId,
            'lastPage'        => $lastPage,
            'progressPercent' => $progressPercent,
        ], 'İlerleme kaydedildi');
    }

    /**
     * GET /api/progress?bookId=...&lang=...
     */
    public function show(array $params): void
    {
        global $currentUser;

        $bookId = Request::queryInt('bookId', 0);
        $lang   = Request::query('lang', 'tr');

        if ($bookId > 0) {
            // Tek kitap ilerlemesi
            $progress = Database::fetch(
                'SELECT rp.*, bt.title
                 FROM reading_progress rp
                 LEFT JOIN book_translations bt ON bt.book_id = rp.book_id AND bt.lang = rp.lang
                 WHERE rp.user_id = ? AND rp.book_id = ? AND rp.lang = ?',
                [$currentUser['id'], $bookId, $lang]
            );

            Response::success($progress);
        } else {
            // Tüm ilerlemeler
            $progresses = Database::fetchAll(
                'SELECT rp.*, bt.title, b.cover_path, b.slug, b.total_pages
                 FROM reading_progress rp
                 INNER JOIN books b ON b.id = rp.book_id
                 LEFT JOIN book_translations bt ON bt.book_id = rp.book_id AND bt.lang = rp.lang
                 WHERE rp.user_id = ?
                 ORDER BY rp.updated_at DESC',
                [$currentUser['id']]
            );

            foreach ($progresses as &$p) {
                $p['coverUrl'] = $p['cover_path'] ? '/api/covers/' . $p['book_id'] : null;
                unset($p['cover_path']);
            }

            Response::success($progresses);
        }
    }
}
