<?php
/**
 * Book Controller - Kitap listeleme, detay, erişim ve içerik
 */
class BookController
{
    /**
     * GET /api/books - Kitap listesi (public)
     */
    public function index(array $params): void
    {
        $config = require __DIR__ . '/../config/app.php';

        $lang  = Request::query('lang', $config['default_language']);
        $page  = max(1, Request::queryInt('page', 1));
        $limit = min($config['pagination']['max_limit'], max(1, Request::queryInt('limit', $config['pagination']['default_limit'])));
        $q     = Request::query('q', '');
        $category = Request::query('category', '');
        $offset = ($page - 1) * $limit;

        $where = ['b.status = ?'];
        $bindings = ['published'];

        // Kategori filtresi
        if ($category) {
            $where[] = 'c.slug = ?';
            $bindings[] = $category;
        }

        // Arama
        if ($q) {
            $where[] = '(bt.title LIKE ? OR bt.author LIKE ? OR bt.description LIKE ? OR bt.tags LIKE ?)';
            $searchTerm = "%{$q}%";
            $bindings = array_merge($bindings, [$searchTerm, $searchTerm, $searchTerm, $searchTerm]);
        }

        $whereClause = implode(' AND ', $where);

        // Toplam sayı
        $countSql = "
            SELECT COUNT(DISTINCT b.id) as total
            FROM books b
            LEFT JOIN book_translations bt ON bt.book_id = b.id AND bt.lang = ?
            LEFT JOIN categories c ON c.id = b.category_id
            WHERE {$whereClause}
        ";
        $total = (int)Database::fetch($countSql, array_merge([$lang], $bindings))['total'];

        // Kitapları getir
        $sql = "
            SELECT
                b.id, b.slug, b.is_free, b.preview_pages, b.total_pages,
                b.cover_path, b.view_count, b.created_at,
                bt.title, bt.author, bt.description, bt.tags,
                c.slug as category_slug,
                ct.name as category_name
            FROM books b
            LEFT JOIN book_translations bt ON bt.book_id = b.id AND bt.lang = ?
            LEFT JOIN categories c ON c.id = b.category_id
            LEFT JOIN category_translations ct ON ct.category_id = c.id AND ct.lang = ?
            WHERE {$whereClause}
            ORDER BY b.sort_order ASC, b.created_at DESC
            LIMIT ? OFFSET ?
        ";

        $books = Database::fetchAll(
            $sql,
            array_merge([$lang, $lang], $bindings, [$limit, $offset])
        );

        // Formatları ekle
        foreach ($books as &$book) {
            $assets = Database::fetchAll(
                'SELECT DISTINCT format FROM book_assets WHERE book_id = ? AND lang = ? AND is_active = 1',
                [$book['id'], $lang]
            );
            $book['availableFormats'] = array_column($assets, 'format');
            $book['coverUrl'] = $book['cover_path'] ? '/api/covers/' . $book['id'] : null;
            unset($book['cover_path']);
        }

        Response::paginated($books, $total, $page, $limit);
    }

    /**
     * GET /api/books/{slug} - Kitap detayı (public)
     */
    public function show(array $params): void
    {
        $config = require __DIR__ . '/../config/app.php';
        $lang = Request::query('lang', $config['default_language']);
        $slug = $params['slug'];

        $book = Database::fetch(
            'SELECT b.*, bt.title, bt.description, bt.author, bt.publisher, bt.isbn,
                    bt.seo_title, bt.seo_desc, bt.tags,
                    c.slug as category_slug, ct.name as category_name
             FROM books b
             LEFT JOIN book_translations bt ON bt.book_id = b.id AND bt.lang = ?
             LEFT JOIN categories c ON c.id = b.category_id
             LEFT JOIN category_translations ct ON ct.category_id = c.id AND ct.lang = ?
             WHERE b.slug = ? AND b.status = ?',
            [$lang, $lang, $slug, 'published']
        );

        if (!$book) {
            Response::error('Kitap bulunamadı', 'BOOK_NOT_FOUND', 404);
        }

        // Görüntüleme sayacı artır
        Database::execute('UPDATE books SET view_count = view_count + 1 WHERE id = ?', [$book['id']]);

        // Mevcut formatlar
        $assets = Database::fetchAll(
            'SELECT format, version, size_bytes FROM book_assets
             WHERE book_id = ? AND lang = ? AND is_active = 1',
            [$book['id'], $lang]
        );

        // Mevcut diller
        $languages = Database::fetchAll(
            'SELECT lang, title FROM book_translations WHERE book_id = ?',
            [$book['id']]
        );

        $book['availableFormats'] = $assets;
        $book['availableLanguages'] = $languages;
        $book['coverUrl'] = $book['cover_path'] ? '/api/covers/' . $book['id'] : null;

        // Hassas alanları kaldır
        unset($book['cover_path'], $book['password_hash']);

        Response::success($book);
    }

    /**
     * GET /api/books/{id}/access - Erişim bilgisi (authenticated)
     */
    public function access(array $params): void
    {
        global $currentUser;
        $config = require __DIR__ . '/../config/app.php';

        $bookId = (int)$params['id'];
        $lang = Request::query('lang', $config['default_language']);

        $book = Database::fetch(
            'SELECT id, is_free, preview_pages, total_pages FROM books WHERE id = ? AND status = ?',
            [$bookId, 'published']
        );

        if (!$book) {
            Response::error('Kitap bulunamadı', 'BOOK_NOT_FOUND', 404);
        }

        // Abonelik durumu kontrol
        $subscriptionActive = $this->isSubscriptionActive($currentUser['id']);

        // Erişim kararı
        $isFree = (bool)$book['is_free'];
        $allowedMaxPage = $book['total_pages']; // varsayılan: tam erişim

        if (!$isFree && !$subscriptionActive) {
            $allowedMaxPage = $book['preview_pages'];
        }

        // Mevcut formatlar
        $assets = Database::fetchAll(
            'SELECT format, version FROM book_assets WHERE book_id = ? AND lang = ? AND is_active = 1',
            [$bookId, $lang]
        );
        $formats = array_column($assets, 'format');

        // Aktif format (chunk öncelikli)
        $activeFormat = 'chunk';
        if (!in_array('chunk', $formats)) {
            $activeFormat = $formats[0] ?? 'chunk';
        }

        // Kullanıcının okuma ilerlemesi
        $progress = Database::fetch(
            'SELECT last_page, last_format, progress_percent, last_location
             FROM reading_progress WHERE user_id = ? AND book_id = ? AND lang = ?',
            [$currentUser['id'], $bookId, $lang]
        );

        Response::success([
            'bookId'             => $bookId,
            'isFree'             => $isFree,
            'subscriptionActive' => $subscriptionActive,
            'previewPages'       => (int)$book['preview_pages'],
            'totalPages'         => (int)$book['total_pages'],
            'allowedMaxPage'     => $allowedMaxPage,
            'availableFormats'   => $formats,
            'activeFormat'       => $activeFormat,
            'progress'           => $progress,
        ]);
    }

    /**
     * GET /api/books/{id}/chunks - Chunk index listesi
     */
    public function chunkIndex(array $params): void
    {
        global $currentUser;
        $config = require __DIR__ . '/../config/app.php';

        $bookId = (int)$params['id'];
        $lang = Request::query('lang', $config['default_language']);
        $version = Request::queryInt('version', 1);

        $book = Database::fetch(
            'SELECT id, is_free, preview_pages, total_pages FROM books WHERE id = ?',
            [$bookId]
        );

        if (!$book) {
            Response::error('Kitap bulunamadı', 'BOOK_NOT_FOUND', 404);
        }

        $chunks = Database::fetchAll(
            'SELECT chunk_no, page_start, page_end
             FROM book_chunk_index
             WHERE book_id = ? AND lang = ? AND version = ?
             ORDER BY chunk_no ASC',
            [$bookId, $lang, $version]
        );

        // Erişim sınırı bilgisi ekle
        $subscriptionActive = $this->isSubscriptionActive($currentUser['id']);
        $allowedMaxPage = $book['total_pages'];

        if (!(bool)$book['is_free'] && !$subscriptionActive) {
            $allowedMaxPage = $book['preview_pages'];
        }

        foreach ($chunks as &$chunk) {
            $chunk['locked'] = $chunk['page_start'] > $allowedMaxPage;
        }

        Response::success([
            'bookId'         => $bookId,
            'lang'           => $lang,
            'version'        => $version,
            'allowedMaxPage' => $allowedMaxPage,
            'chunks'         => $chunks,
        ], 'OK', 200);
    }

    /**
     * GET /api/books/{id}/chunk/{chunkNo} - Tek chunk içeriği
     */
    public function chunk(array $params): void
    {
        global $currentUser;
        $config = require __DIR__ . '/../config/app.php';

        $bookId = (int)$params['id'];
        $chunkNo = (int)$params['chunkNo'];
        $lang = Request::query('lang', $config['default_language']);
        $version = Request::queryInt('version', 1);

        $book = Database::fetch(
            'SELECT id, is_free, preview_pages, total_pages FROM books WHERE id = ?',
            [$bookId]
        );

        if (!$book) {
            Response::error('Kitap bulunamadı', 'BOOK_NOT_FOUND', 404);
        }

        // Chunk meta bilgisini al
        $chunkMeta = Database::fetch(
            'SELECT * FROM book_chunk_index
             WHERE book_id = ? AND lang = ? AND version = ? AND chunk_no = ?',
            [$bookId, $lang, $version, $chunkNo]
        );

        if (!$chunkMeta) {
            Response::error('Chunk bulunamadı', 'CHUNK_NOT_FOUND', 404);
        }

        // Erişim kontrolü
        $subscriptionActive = $this->isSubscriptionActive($currentUser['id']);
        $isFree = (bool)$book['is_free'];
        $allowedMaxPage = $book['total_pages'];

        if (!$isFree && !$subscriptionActive) {
            $allowedMaxPage = $book['preview_pages'];
        }

        // Chunk tamamen kilitli mi?
        if ($chunkMeta['page_start'] > $allowedMaxPage) {
            Response::error(
                'Bu içeriğe erişmek için abonelik gerekli',
                'PAYWALL_REQUIRED',
                403,
                ['allowedMaxPage' => $allowedMaxPage]
            );
        }

        // Chunk dosyasını oku
        $storagePath = $config['storage_path'];
        $filePath = $storagePath . '/' . $chunkMeta['file_path'];

        if (!file_exists($filePath)) {
            Response::error('Chunk dosyası bulunamadı', 'FILE_NOT_FOUND', 500);
        }

        $content = json_decode(file_get_contents($filePath), true);

        if (!$content) {
            Response::error('Chunk dosyası okunamadı', 'FILE_READ_ERROR', 500);
        }

        // Eğer chunk kısmen erişilebilirse, sayfaları kırp
        if (!$isFree && !$subscriptionActive && $chunkMeta['page_end'] > $allowedMaxPage) {
            if (isset($content['pages'])) {
                $content['pages'] = array_filter(
                    $content['pages'],
                    fn($p) => $p['pageNo'] <= $allowedMaxPage
                );
                $content['pages'] = array_values($content['pages']);
            }
            $content['truncated'] = true;
            $content['allowedMaxPage'] = $allowedMaxPage;
        }

        // Cache header
        $cacheTime = $config['cache']['chunk_content'];
        header("Cache-Control: public, max-age={$cacheTime}");

        Response::success($content);
    }

    /**
     * GET /api/books/{id}/download - EPUB/PDF indirme
     */
    public function download(array $params): void
    {
        global $currentUser;
        $config = require __DIR__ . '/../config/app.php';

        $bookId = (int)$params['id'];
        $lang = Request::query('lang', $config['default_language']);
        $format = Request::query('format', 'epub');

        if (!in_array($format, ['epub', 'pdf'])) {
            Response::error('Geçersiz format. epub veya pdf olmalı', 'INVALID_FORMAT', 400);
        }

        $book = Database::fetch(
            'SELECT id, is_free, preview_pages FROM books WHERE id = ? AND status = ?',
            [$bookId, 'published']
        );

        if (!$book) {
            Response::error('Kitap bulunamadı', 'BOOK_NOT_FOUND', 404);
        }

        // Asset bilgisini al
        $asset = Database::fetch(
            'SELECT path, preview_path FROM book_assets
             WHERE book_id = ? AND lang = ? AND format = ? AND is_active = 1
             ORDER BY version DESC LIMIT 1',
            [$bookId, $lang, $format]
        );

        if (!$asset) {
            Response::error('Bu formatta içerik bulunamadı', 'ASSET_NOT_FOUND', 404);
        }

        $storagePath = $config['storage_path'];
        $subscriptionActive = $this->isSubscriptionActive($currentUser['id']);
        $isFree = (bool)$book['is_free'];

        // Erişim kararı: tam dosya mı, preview mi?
        if ($isFree || $subscriptionActive) {
            $filePath = $storagePath . '/' . $asset['path'];
        } else {
            // Preview dosyası
            if ($asset['preview_path']) {
                $filePath = $storagePath . '/' . $asset['preview_path'];
            } else {
                Response::error(
                    'Preview dosyası mevcut değil. Abonelik gerekli.',
                    'PAYWALL_REQUIRED',
                    403
                );
            }
        }

        if (!file_exists($filePath)) {
            Response::error('Dosya bulunamadı', 'FILE_NOT_FOUND', 500);
        }

        $mimeType = $format === 'epub'
            ? 'application/epub+zip'
            : 'application/pdf';
        $filename = "book_{$bookId}_{$lang}.{$format}";

        Response::file($filePath, $filename, $mimeType);
    }

    /**
     * Kullanıcının aktif aboneliği var mı?
     */
    private function isSubscriptionActive(int $userId): bool
    {
        $sub = Database::fetch(
            'SELECT id FROM subscriptions
             WHERE user_id = ? AND is_active = 1 AND expiry_time > NOW()
             LIMIT 1',
            [$userId]
        );
        return $sub !== null;
    }
}
