<?php
/**
 * Auth Controller - Kimlik doğrulama işlemleri
 */
class AuthController
{
    /**
     * POST /api/auth/register
     */
    public function register(array $params): void
    {
        $data = Request::validate(['email', 'password', 'name']);

        $email = strtolower(trim($data['email']));
        $password = $data['password'];
        $name = trim($data['name']);
        $locale = Request::input('locale', 'tr');

        // Email format kontrolü
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            Response::error('Geçersiz e-posta adresi', 'INVALID_EMAIL', 422);
        }

        // Şifre uzunluk kontrolü
        if (strlen($password) < 6) {
            Response::error('Şifre en az 6 karakter olmalıdır', 'WEAK_PASSWORD', 422);
        }

        // Email tekrarı kontrolü
        $exists = Database::fetch('SELECT id FROM users WHERE email = ?', [$email]);
        if ($exists) {
            Response::error('Bu e-posta adresi zaten kayıtlı', 'EMAIL_EXISTS', 409);
        }

        // Kullanıcı oluştur
        $passwordHash = password_hash($password, PASSWORD_BCRYPT);
        $userId = Database::insert(
            'INSERT INTO users (email, password_hash, name, locale) VALUES (?, ?, ?, ?)',
            [$email, $passwordHash, $name, $locale]
        );

        // Token'ları oluştur
        $tokens = $this->generateTokens((int)$userId);

        $user = Database::fetch('SELECT id, email, name, locale, created_at FROM users WHERE id = ?', [$userId]);

        Response::success([
            'user'         => $user,
            'accessToken'  => $tokens['accessToken'],
            'refreshToken' => $tokens['refreshToken'],
            'expiresIn'    => $tokens['expiresIn'],
        ], 'Kayıt başarılı', 201);
    }

    /**
     * POST /api/auth/login
     */
    public function login(array $params): void
    {
        $data = Request::validate(['email', 'password']);

        $email = strtolower(trim($data['email']));
        $password = $data['password'];

        $user = Database::fetch('SELECT * FROM users WHERE email = ?', [$email]);

        if (!$user || !password_verify($password, $user['password_hash'])) {
            Response::error('E-posta veya şifre hatalı', 'INVALID_CREDENTIALS', 401);
        }

        if (!$user['is_active']) {
            Response::error('Hesabınız devre dışı bırakılmış', 'ACCOUNT_DISABLED', 403);
        }

        // Son giriş zamanını güncelle
        Database::execute('UPDATE users SET last_login_at = NOW() WHERE id = ?', [$user['id']]);

        // Token'ları oluştur
        $tokens = $this->generateTokens((int)$user['id']);

        // Rolleri al
        $roles = Database::fetchAll(
            'SELECT r.name FROM roles r INNER JOIN user_roles ur ON ur.role_id = r.id WHERE ur.user_id = ?',
            [$user['id']]
        );

        Response::success([
            'user' => [
                'id'        => $user['id'],
                'email'     => $user['email'],
                'name'      => $user['name'],
                'locale'    => $user['locale'],
                'roles'     => array_column($roles, 'name'),
                'createdAt' => $user['created_at'],
            ],
            'accessToken'  => $tokens['accessToken'],
            'refreshToken' => $tokens['refreshToken'],
            'expiresIn'    => $tokens['expiresIn'],
        ], 'Giriş başarılı');
    }

    /**
     * POST /api/auth/refresh
     */
    public function refresh(array $params): void
    {
        $refreshToken = Request::input('refreshToken');
        if (!$refreshToken) {
            Response::error('Refresh token gerekli', 'MISSING_TOKEN', 400);
        }

        // Token'ı doğrula
        $payload = JWT::verify($refreshToken, 'refresh');
        if (!$payload) {
            Response::error('Refresh token geçersiz veya süresi dolmuş', 'INVALID_REFRESH_TOKEN', 401);
        }

        // DB'de token hash kontrolü
        $tokenHash = hash('sha256', $refreshToken);
        $stored = Database::fetch(
            'SELECT * FROM refresh_tokens WHERE token_hash = ? AND user_id = ? AND expires_at > NOW()',
            [$tokenHash, $payload['sub']]
        );

        if (!$stored) {
            Response::error('Refresh token geçersiz', 'INVALID_REFRESH_TOKEN', 401);
        }

        // Eski token'ı sil
        Database::execute('DELETE FROM refresh_tokens WHERE id = ?', [$stored['id']]);

        // Yeni token'lar oluştur
        $tokens = $this->generateTokens((int)$payload['sub']);

        Response::success([
            'accessToken'  => $tokens['accessToken'],
            'refreshToken' => $tokens['refreshToken'],
            'expiresIn'    => $tokens['expiresIn'],
        ], 'Token yenilendi');
    }

    /**
     * POST /api/auth/logout
     */
    public function logout(array $params): void
    {
        global $currentUser;

        // Tüm refresh token'ları sil
        Database::execute('DELETE FROM refresh_tokens WHERE user_id = ?', [$currentUser['id']]);

        Response::success(null, 'Çıkış yapıldı');
    }

    /**
     * GET /api/me
     */
    public function me(array $params): void
    {
        global $currentUser;

        // Abonelik durumunu al
        $subscription = Database::fetch(
            'SELECT product_id, is_active, expiry_time, auto_renewing FROM subscriptions
             WHERE user_id = ? AND is_active = 1 ORDER BY expiry_time DESC LIMIT 1',
            [$currentUser['id']]
        );

        // Okuma istatistikleri
        $stats = Database::fetch(
            'SELECT COUNT(*) as books_read, SUM(total_reading_time) as total_time
             FROM reading_progress WHERE user_id = ?',
            [$currentUser['id']]
        );

        Response::success([
            'user' => [
                'id'     => $currentUser['id'],
                'email'  => $currentUser['email'],
                'name'   => $currentUser['name'],
                'locale' => $currentUser['locale'],
                'roles'  => $currentUser['roles'],
            ],
            'subscription' => $subscription ? [
                'productId'    => $subscription['product_id'],
                'isActive'     => (bool)$subscription['is_active'],
                'expiryTime'   => $subscription['expiry_time'],
                'autoRenewing' => (bool)$subscription['auto_renewing'],
            ] : null,
            'stats' => [
                'booksRead'        => (int)($stats['books_read'] ?? 0),
                'totalReadingTime' => (int)($stats['total_time'] ?? 0),
            ],
        ]);
    }

    /**
     * Token çifti oluştur ve refresh token'ı DB'ye kaydet
     */
    private function generateTokens(int $userId): array
    {
        $accessToken = JWT::createAccessToken(['sub' => $userId]);
        $refreshToken = JWT::createRefreshToken(['sub' => $userId]);

        $config = require __DIR__ . '/../config/app.php';

        // Refresh token'ı DB'ye kaydet
        $tokenHash = hash('sha256', $refreshToken);
        $expiresAt = date('Y-m-d H:i:s', time() + $config['jwt']['refresh_ttl']);

        Database::insert(
            'INSERT INTO refresh_tokens (user_id, token_hash, expires_at) VALUES (?, ?, ?)',
            [$userId, $tokenHash, $expiresAt]
        );

        return [
            'accessToken'  => $accessToken,
            'refreshToken' => $refreshToken,
            'expiresIn'    => $config['jwt']['access_ttl'],
        ];
    }
}
