<?php
/**
 * Admin User Controller - Kullanıcı ve rol yönetimi
 */
class AdminUserController
{
    /**
     * GET /api/admin/users
     */
    public function index(array $params): void
    {
        $page  = max(1, Request::queryInt('page', 1));
        $limit = min(100, max(1, Request::queryInt('limit', 20)));
        $offset = ($page - 1) * $limit;
        $q = Request::query('q', '');

        $where = ['1=1'];
        $bindings = [];

        if ($q) {
            $where[] = '(u.email LIKE ? OR u.name LIKE ?)';
            $bindings[] = "%{$q}%";
            $bindings[] = "%{$q}%";
        }

        $whereClause = implode(' AND ', $where);

        $total = (int)Database::fetch(
            "SELECT COUNT(*) as c FROM users u WHERE {$whereClause}",
            $bindings
        )['c'];

        $users = Database::fetchAll(
            "SELECT u.id, u.email, u.name, u.locale, u.is_active, u.last_login_at, u.created_at
             FROM users u WHERE {$whereClause}
             ORDER BY u.created_at DESC
             LIMIT ? OFFSET ?",
            array_merge($bindings, [$limit, $offset])
        );

        foreach ($users as &$user) {
            $roles = Database::fetchAll(
                'SELECT r.name FROM roles r
                 INNER JOIN user_roles ur ON ur.role_id = r.id
                 WHERE ur.user_id = ?',
                [$user['id']]
            );
            $user['roles'] = array_column($roles, 'name');

            $sub = Database::fetch(
                'SELECT product_id, is_active, expiry_time FROM subscriptions
                 WHERE user_id = ? ORDER BY expiry_time DESC LIMIT 1',
                [$user['id']]
            );
            $user['subscription'] = $sub;
        }

        Response::paginated($users, $total, $page, $limit);
    }

    /**
     * GET /api/admin/users/{id}
     */
    public function show(array $params): void
    {
        $id = (int)$params['id'];

        $user = Database::fetch(
            'SELECT id, email, name, locale, is_active, avatar_url, last_login_at, created_at, updated_at
             FROM users WHERE id = ?',
            [$id]
        );

        if (!$user) {
            Response::error('Kullanıcı bulunamadı', 'USER_NOT_FOUND', 404);
        }

        $roles = Database::fetchAll(
            'SELECT r.id, r.name, r.description FROM roles r
             INNER JOIN user_roles ur ON ur.role_id = r.id
             WHERE ur.user_id = ?',
            [$id]
        );
        $user['roles'] = $roles;

        $subscriptions = Database::fetchAll(
            'SELECT * FROM subscriptions WHERE user_id = ? ORDER BY created_at DESC LIMIT 10',
            [$id]
        );
        $user['subscriptions'] = $subscriptions;

        $progress = Database::fetchAll(
            'SELECT rp.*, bt.title
             FROM reading_progress rp
             LEFT JOIN book_translations bt ON bt.book_id = rp.book_id AND bt.lang = rp.lang
             WHERE rp.user_id = ?
             ORDER BY rp.updated_at DESC LIMIT 20',
            [$id]
        );
        $user['readingProgress'] = $progress;

        Response::success($user);
    }

    /**
     * PUT /api/admin/users/{id}/roles
     * Body: { "roles": ["admin", "editor"] }
     */
    public function updateRoles(array $params): void
    {
        global $currentUser;
        $id = (int)$params['id'];
        $data = Request::validate(['roles']);

        $user = Database::fetch('SELECT id FROM users WHERE id = ?', [$id]);
        if (!$user) {
            Response::error('Kullanıcı bulunamadı', 'USER_NOT_FOUND', 404);
        }

        // superAdmin rolü sadece mevcut superAdmin tarafından verilebilir
        $requestedRoles = $data['roles'];
        if (in_array('superAdmin', $requestedRoles) && !in_array('superAdmin', $currentUser['roles'])) {
            Response::error('superAdmin rolü atama yetkiniz yok', 'FORBIDDEN', 403);
        }

        // Mevcut rolleri temizle
        Database::execute('DELETE FROM user_roles WHERE user_id = ?', [$id]);

        // Yeni rolleri ekle
        foreach ($requestedRoles as $roleName) {
            $role = Database::fetch('SELECT id FROM roles WHERE name = ?', [$roleName]);
            if ($role) {
                Database::insert(
                    'INSERT INTO user_roles (user_id, role_id) VALUES (?, ?)',
                    [$id, $role['id']]
                );
            }
        }

        Response::success(null, 'Roller güncellendi');
    }
}
