<?php
/**
 * Admin Asset Controller - Dosya yükleme (Chunk/EPUB/PDF)
 */
class AdminAssetController
{
    /**
     * GET /api/admin/books/{id}/assets
     */
    public function index(array $params): void
    {
        $bookId = (int)$params['id'];

        $assets = Database::fetchAll(
            'SELECT * FROM book_assets WHERE book_id = ? ORDER BY lang, format, version DESC',
            [$bookId]
        );

        $chunks = Database::fetchAll(
            'SELECT * FROM book_chunk_index WHERE book_id = ? ORDER BY lang, version, chunk_no',
            [$bookId]
        );

        Response::success([
            'assets' => $assets,
            'chunks' => $chunks,
        ]);
    }

    /**
     * POST /api/admin/books/{id}/assets/upload
     * Multipart form: file, lang, format (epub|pdf), isPreview (0|1)
     */
    public function upload(array $params): void
    {
        $config = require __DIR__ . '/../config/app.php';
        $bookId = (int)$params['id'];

        // Kitap var mı?
        if (!Database::fetch('SELECT id FROM books WHERE id = ?', [$bookId])) {
            Response::error('Kitap bulunamadı', 'BOOK_NOT_FOUND', 404);
        }

        $file = Request::file('file');
        if (!$file || $file['error'] !== UPLOAD_ERR_OK) {
            Response::error('Dosya yükleme hatası', 'UPLOAD_ERROR', 400);
        }

        $lang = $_POST['lang'] ?? 'tr';
        $format = $_POST['format'] ?? 'epub';
        $isPreview = (bool)($_POST['isPreview'] ?? false);
        $version = (int)($_POST['version'] ?? 1);

        if (!in_array($format, ['epub', 'pdf'])) {
            Response::error('Format epub veya pdf olmalı', 'INVALID_FORMAT', 400);
        }

        // Dosya uzantısı kontrolü
        $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if ($ext !== $format) {
            Response::error("Dosya uzantısı {$format} olmalı", 'INVALID_EXTENSION', 400);
        }

        // Hedef klasör
        $storagePath = $config['storage_path'];
        $dir = "{$storagePath}/books/{$bookId}/{$lang}/{$format}/v{$version}";
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }

        $filename = $isPreview ? "preview.{$format}" : "full.{$format}";
        $targetPath = "{$dir}/{$filename}";

        if (!move_uploaded_file($file['tmp_name'], $targetPath)) {
            Response::error('Dosya kaydedilemedi', 'SAVE_ERROR', 500);
        }

        $relativePath = "books/{$bookId}/{$lang}/{$format}/v{$version}/{$filename}";
        $checksum = hash_file('sha256', $targetPath);
        $size = filesize($targetPath);

        // Asset kaydı oluştur/güncelle
        $existing = Database::fetch(
            'SELECT id, path, preview_path FROM book_assets
             WHERE book_id = ? AND lang = ? AND format = ? AND version = ?',
            [$bookId, $lang, $format, $version]
        );

        if ($existing) {
            if ($isPreview) {
                Database::execute(
                    'UPDATE book_assets SET preview_path = ?, checksum = ?, size_bytes = ? WHERE id = ?',
                    [$relativePath, $checksum, $size, $existing['id']]
                );
            } else {
                Database::execute(
                    'UPDATE book_assets SET path = ?, checksum = ?, size_bytes = ?, is_active = 1 WHERE id = ?',
                    [$relativePath, $checksum, $size, $existing['id']]
                );
            }
        } else {
            Database::insert(
                'INSERT INTO book_assets (book_id, lang, format, version, path, preview_path, is_active, size_bytes, checksum)
                 VALUES (?, ?, ?, ?, ?, ?, 1, ?, ?)',
                [
                    $bookId, $lang, $format, $version,
                    $isPreview ? '' : $relativePath,
                    $isPreview ? $relativePath : null,
                    $size, $checksum
                ]
            );
        }

        Response::success([
            'path'     => $relativePath,
            'size'     => $size,
            'checksum' => $checksum,
        ], 'Dosya yüklendi', 201);
    }

    /**
     * POST /api/admin/books/{id}/chunks/upload
     * Multipart form: files[] (JSON dosyaları), lang, version
     */
    public function uploadChunks(array $params): void
    {
        $config = require __DIR__ . '/../config/app.php';
        $bookId = (int)$params['id'];

        if (!Database::fetch('SELECT id FROM books WHERE id = ?', [$bookId])) {
            Response::error('Kitap bulunamadı', 'BOOK_NOT_FOUND', 404);
        }

        $lang = $_POST['lang'] ?? 'tr';
        $version = (int)($_POST['version'] ?? 1);

        // Hedef klasör
        $storagePath = $config['storage_path'];
        $dir = "{$storagePath}/books/{$bookId}/{$lang}/chunk/v{$version}";
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }

        // Eski chunk index'i temizle
        Database::execute(
            'DELETE FROM book_chunk_index WHERE book_id = ? AND lang = ? AND version = ?',
            [$bookId, $lang, $version]
        );

        $uploaded = [];
        $files = $_FILES['files'] ?? null;

        if (!$files) {
            // Tek dosya olarak da dene
            $files = $_FILES['file'] ?? null;
        }

        if (!$files) {
            Response::error('Chunk dosyaları gerekli', 'NO_FILES', 400);
        }

        // Birden fazla dosya normalize
        if (is_array($files['name'])) {
            $fileCount = count($files['name']);
            for ($i = 0; $i < $fileCount; $i++) {
                if ($files['error'][$i] !== UPLOAD_ERR_OK) continue;

                $result = $this->processChunkFile(
                    $files['tmp_name'][$i],
                    $files['name'][$i],
                    $bookId, $lang, $version, $dir, $storagePath
                );
                if ($result) $uploaded[] = $result;
            }
        } else {
            if ($files['error'] === UPLOAD_ERR_OK) {
                $result = $this->processChunkFile(
                    $files['tmp_name'],
                    $files['name'],
                    $bookId, $lang, $version, $dir, $storagePath
                );
                if ($result) $uploaded[] = $result;
            }
        }

        // Asset kaydı oluştur
        $existing = Database::fetch(
            'SELECT id FROM book_assets WHERE book_id = ? AND lang = ? AND format = ? AND version = ?',
            [$bookId, $lang, 'chunk', $version]
        );

        $basePath = "books/{$bookId}/{$lang}/chunk/v{$version}";
        if ($existing) {
            Database::execute(
                'UPDATE book_assets SET path = ?, is_active = 1 WHERE id = ?',
                [$basePath, $existing['id']]
            );
        } else {
            Database::insert(
                'INSERT INTO book_assets (book_id, lang, format, version, path, is_active) VALUES (?, ?, ?, ?, ?, 1)',
                [$bookId, $lang, 'chunk', $version, $basePath]
            );
        }

        Response::success([
            'uploaded' => $uploaded,
            'count'    => count($uploaded),
        ], 'Chunk dosyaları yüklendi', 201);
    }

    /**
     * Tek bir chunk dosyasını işle
     */
    private function processChunkFile(
        string $tmpPath, string $originalName,
        int $bookId, string $lang, int $version,
        string $dir, string $storagePath
    ): ?array {
        $content = file_get_contents($tmpPath);
        $data = json_decode($content, true);

        if (!$data) return null;

        $chunkNo = $data['chunkNo'] ?? null;
        $pageStart = $data['pageStart'] ?? null;
        $pageEnd = $data['pageEnd'] ?? null;

        if ($chunkNo === null) {
            // Dosya adından çıkar: chunk_001.json
            if (preg_match('/chunk_(\d+)/', $originalName, $m)) {
                $chunkNo = (int)$m[1];
            } else {
                return null;
            }
        }

        $filename = sprintf('chunk_%03d.json', $chunkNo);
        $targetPath = "{$dir}/{$filename}";
        $relativePath = "books/{$bookId}/{$lang}/chunk/v{$version}/{$filename}";

        // Dosyayı kaydet
        if (!copy($tmpPath, $targetPath)) {
            return null;
        }

        // Chunk index kaydı
        Database::insert(
            'INSERT INTO book_chunk_index (book_id, lang, version, chunk_no, page_start, page_end, file_path)
             VALUES (?, ?, ?, ?, ?, ?, ?)',
            [$bookId, $lang, $version, $chunkNo, $pageStart ?? 0, $pageEnd ?? 0, $relativePath]
        );

        return [
            'chunkNo'   => $chunkNo,
            'pageStart' => $pageStart,
            'pageEnd'   => $pageEnd,
            'file'      => $filename,
        ];
    }

    /**
     * DELETE /api/admin/assets/{id}
     */
    public function destroy(array $params): void
    {
        $config = require __DIR__ . '/../config/app.php';
        $id = (int)$params['id'];

        $asset = Database::fetch('SELECT * FROM book_assets WHERE id = ?', [$id]);
        if (!$asset) {
            Response::error('Asset bulunamadı', 'ASSET_NOT_FOUND', 404);
        }

        // Dosyaları sil
        $storagePath = $config['storage_path'];
        if ($asset['path'] && file_exists("{$storagePath}/{$asset['path']}")) {
            if ($asset['format'] === 'chunk') {
                // Chunk dizinini sil
                $dir = "{$storagePath}/{$asset['path']}";
                if (is_dir($dir)) {
                    $files = glob("{$dir}/*.json");
                    foreach ($files as $f) unlink($f);
                    rmdir($dir);
                }
            } else {
                unlink("{$storagePath}/{$asset['path']}");
            }
        }
        if ($asset['preview_path'] && file_exists("{$storagePath}/{$asset['preview_path']}")) {
            unlink("{$storagePath}/{$asset['preview_path']}");
        }

        // Chunk index temizle
        if ($asset['format'] === 'chunk') {
            Database::execute(
                'DELETE FROM book_chunk_index WHERE book_id = ? AND lang = ? AND version = ?',
                [$asset['book_id'], $asset['lang'], $asset['version']]
            );
        }

        Database::execute('DELETE FROM book_assets WHERE id = ?', [$id]);

        Response::success(null, 'Asset silindi');
    }
}
