<?php
/**
 * E-Kitap API Debug/Test Scripti
 * Bu dosyayı tarayıcıdan açarak API'nin çalışıp çalışmadığını test edin:
 * https://ekitap.2w.tr/backend/api/test.php
 *
 * ÖNEMLİ: Test sonrası bu dosyayı silin!
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');

$results = [];

// 1. PHP Versiyonu
$results['php_version'] = PHP_VERSION;
$results['php_version_ok'] = version_compare(PHP_VERSION, '8.0.0', '>=');

// 2. Gerekli PHP Uzantıları
$extensions = ['pdo', 'pdo_mysql', 'json', 'mbstring', 'openssl'];
$results['extensions'] = [];
foreach ($extensions as $ext) {
    $results['extensions'][$ext] = extension_loaded($ext);
}

// 3. Dosya Yapısı Kontrolü
$backendDir = dirname(__DIR__);
$requiredFiles = [
    'core/Router.php',
    'core/Response.php',
    'core/Database.php',
    'core/Request.php',
    'core/JWT.php',
    'middleware/RateLimitMiddleware.php',
    'middleware/AuthMiddleware.php',
    'middleware/AdminMiddleware.php',
    'controllers/AuthController.php',
    'controllers/BookController.php',
    'config/database.php',
    'config/app.php',
    'routes/api.php',
    '.env',
];
$results['files'] = [];
foreach ($requiredFiles as $f) {
    $results['files'][$f] = file_exists($backendDir . '/' . $f);
}

// 4. .env dosyası yüklenebiliyor mu?
$results['env_loaded'] = false;
$envFile = $backendDir . '/.env';
if (file_exists($envFile)) {
    $lines = file($envFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    $envVars = [];
    foreach ($lines as $line) {
        if (str_starts_with(trim($line), '#')) continue;
        if (strpos($line, '=') === false) continue;
        [$key, $value] = explode('=', $line, 2);
        $envVars[trim($key)] = '***'; // Güvenlik için değeri gösterme
    }
    $results['env_loaded'] = true;
    $results['env_keys'] = array_keys($envVars);
}

// 5. Veritabanı bağlantısı
$results['database'] = ['connected' => false];
try {
    $config = require $backendDir . '/config/database.php';
    $dsn = "mysql:host={$config['host']};dbname={$config['dbname']};charset={$config['charset']}";
    $pdo = new PDO($dsn, $config['username'], $config['password'], [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    ]);
    $results['database']['connected'] = true;

    // Tabloları kontrol et
    $tables = $pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
    $results['database']['tables'] = $tables;
    $results['database']['table_count'] = count($tables);
} catch (PDOException $e) {
    $results['database']['error'] = $e->getMessage();
}

// 6. REQUEST_URI kontrolü (routing debug)
$results['server'] = [
    'REQUEST_URI' => $_SERVER['REQUEST_URI'] ?? 'N/A',
    'SCRIPT_NAME' => $_SERVER['SCRIPT_NAME'] ?? 'N/A',
    'SCRIPT_FILENAME' => $_SERVER['SCRIPT_FILENAME'] ?? 'N/A',
    'DOCUMENT_ROOT' => $_SERVER['DOCUMENT_ROOT'] ?? 'N/A',
    'PHP_SELF' => $_SERVER['PHP_SELF'] ?? 'N/A',
    'REQUEST_METHOD' => $_SERVER['REQUEST_METHOD'] ?? 'N/A',
    'HTTP_HOST' => $_SERVER['HTTP_HOST'] ?? 'N/A',
];

// 7. Dizin izinleri
$results['permissions'] = [];
$dirsToCheck = [
    'backend' => $backendDir,
    'storage' => dirname($backendDir) . '/storage',
    'storage/books' => dirname($backendDir) . '/storage/books',
    'storage/covers' => dirname($backendDir) . '/storage/covers',
];
foreach ($dirsToCheck as $name => $path) {
    $results['permissions'][$name] = [
        'exists' => is_dir($path),
        'writable' => is_writable($path),
    ];
}

// 8. Autoloader testi
$results['autoloader'] = ['works' => false];
try {
    spl_autoload_register(function (string $class) use ($backendDir) {
        $directories = [
            $backendDir . '/core/',
            $backendDir . '/middleware/',
            $backendDir . '/controllers/',
            $backendDir . '/models/',
        ];
        foreach ($directories as $dir) {
            $file = $dir . $class . '.php';
            if (file_exists($file)) {
                require_once $file;
                return;
            }
        }
    });

    // Router class'ını yüklemeyi dene
    $router = new Router();
    $results['autoloader']['works'] = true;
    $results['autoloader']['router_loaded'] = true;

    // Route'ları yüklemeyi dene
    $registerRoutes = require $backendDir . '/routes/api.php';
    $registerRoutes($router);
    $results['autoloader']['routes_loaded'] = true;

} catch (Throwable $e) {
    $results['autoloader']['error'] = $e->getMessage();
    $results['autoloader']['error_file'] = basename($e->getFile()) . ':' . $e->getLine();
}

// Sonuçları göster
$results['test_time'] = date('Y-m-d H:i:s');
$results['summary'] = 'Tüm testler tamamlandı';

echo json_encode($results, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
